<?php
file_put_contents(__DIR__.'/mastodon_access.log', date('c').' '.$_SERVER['REMOTE_ADDR'].' '.$_SERVER['HTTP_USER_AGENT']."\n", FILE_APPEND);



/*  Paginated public outbox (5 activities / page)
 *  – Image URLs in post body are moved to ActivityStreams attachments
 *  – Disabled User-Agent/IP restrictions for Mastodon fetch testing
 *  – Added debug info & logging
 *  – 1-week HTTP cache
 */

header('Content-Type: application/activity+json; charset=utf-8');
header('Cache-Control: public, max-age=604800'); // 1 week

/* ─────────────────────  CONFIG  ───────────────────── */
$perPage         = 5;
$jsonFile        = __DIR__ . '/data_alcea.json';
$logFile         = __DIR__ . '/outbox_debug.log';  // debug log
/* ───────────────────────────────────────────────────── */

/* ──  Debug logging helper  ────────────────────────── */
function debug_log($msg) {
    global $logFile;
    $time = date('Y-m-d H:i:s');
    file_put_contents($logFile, "[$time] $msg\n", FILE_APPEND);
}

/* ──  Access-control: DISABLED to allow all fetches for debugging ── */
/*
$allowedAgents   = ['mastodon.social', 'mas.to', 'mstdn.jp'];
$allowedIPs      = ['109.43.50.174'];                      
$ua      = $_SERVER['HTTP_USER_AGENT'] ?? '';
$ip      = $_SERVER['REMOTE_ADDR']     ?? '';
$allowed = in_array($ip, $allowedIPs, true);

foreach ($allowedAgents as $tag) {
    if (stripos($ua, $tag) !== false) { $allowed = true; break; }
}

if (!$allowed) {
    http_response_code(403);
    echo json_encode([
        'error' => 'Access denied',
        'debug' => [
            'required_user_agents' => $allowedAgents,
            'required_ips'        => $allowedIPs,
            'your_user_agent'     => $ua,
            'your_ip'             => $ip
        ]
    ], JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    debug_log("Access denied: UA=$ua IP=$ip");
    exit;
}
debug_log("Access granted: UA={$_SERVER['HTTP_USER_AGENT']} IP={$_SERVER['REMOTE_ADDR']}");
*/

/* ──  Environment / paths  ─────────────────────────── */
$scheme  = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
$host    = $_SERVER['HTTP_HOST'];
$baseURL = "$scheme://$host/fedi/outbox.php";
$actor   = "$scheme://$host/@alcea";
/* ───────────────────────────────────────────────────── */

/* ──  Load & flatten JSON  ─────────────────────────── */
if (!file_exists($jsonFile)) {
    http_response_code(500); 
    $err = json_encode(['error' => 'Data file missing'], JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    debug_log("Error: Data file missing");
    echo $err;
    exit;
}
$data  = json_decode(file_get_contents($jsonFile), true);
if (!is_array($data)) {
    http_response_code(500);
    $err = json_encode(['error' => 'Invalid JSON data'], JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    debug_log("Error: Invalid JSON data");
    echo $err;
    exit;
}

$posts = [];
foreach ($data as $entry) {
    foreach ($entry as $date => $info) {
        $posts[] = [
            'date'     => $date,
            'content'  => $info['value'],
            'hashtags' => isset($info['hashtags']) ? array_map('trim', explode(',', $info['hashtags'])) : []
        ];
    }
}
usort($posts, fn($a, $b) => strcmp($b['date'], $a['date']));
$total = count($posts);
debug_log("Loaded $total posts");

/* ──  Pagination  ───────────────────────────────────── */
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : null;
if ($page === null) {
    $resp = [
        '@context'    => 'https://www.w3.org/ns/activitystreams',
        'id'          => $baseURL,
        'type'        => 'OrderedCollection',
        'totalItems'  => $total,
        'first'       => "$baseURL?page=1"
    ];
    debug_log("Returned OrderedCollection metadata");
    echo json_encode($resp, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit;
}

$start  = ($page - 1) * $perPage;
$chunk  = array_slice($posts, $start, $perPage);
debug_log("Serving page $page, items " . count($chunk));

/* ──  Helpers  ──────────────────────────────────────── */
function extractImages(string &$text): array {
    $pattern = '~https?://[^\s"<>]+\.(?:jpe?g|png|gif|webp)~i';
    preg_match_all($pattern, $text, $m);
    foreach ($m[0] ?? [] as $url) $text = str_replace($url, '', $text);
    $text = trim(preg_replace('/\s+/', ' ', $text));
    return $m[0] ?? [];
}
function mimeGuess(string $u): string {
    $e = strtolower(pathinfo(parse_url($u, PHP_URL_PATH), PATHINFO_EXTENSION));
    return ['jpg'=>'image/jpeg','jpeg'=>'image/jpeg','png'=>'image/png','gif'=>'image/gif','webp'=>'image/webp'][$e] ?? 'image/*';
}
/* ───────────────────────────────────────────────────── */

/* ──  Build activities  ─────────────────────────────── */
$items = [];
foreach ($chunk as $p) {
    $id      = "$scheme://$host/fedi/post/{$p['date']}";
    $content = $p['content'];
    $imgs    = extractImages($content);

    $note = [
        'id'            => $id,
        'type'          => 'Note',
        'attributedTo'  => $actor,
        'content'       => $content,
        'published'     => date('c', strtotime($p['date'])),
        'to'            => ['https://www.w3.org/ns/activitystreams#Public']
    ];
    if ($p['hashtags'])
        $note['tag'] = array_map(fn($t)=>['type'=>'Hashtag','name'=>'#'.ltrim($t,'#')], $p['hashtags']);
    if ($imgs)
        $note['attachment'] = array_map(fn($u)=>['type'=>'Image','mediaType'=>mimeGuess($u),'url'=>$u], $imgs);

    $items[] = [
        '@context'  => 'https://www.w3.org/ns/activitystreams',
        'id'        => "$id/create",
        'type'      => 'Create',
        'actor'     => $actor,
        'published' => date('c', strtotime($p['date'])),
        'to'        => ['https://www.w3.org/ns/activitystreams#Public'],
        'object'    => $note
    ];
}
debug_log("Built " . count($items) . " activities");

/* ──  Output page  ──────────────────────────────────── */
$pageObj = [
    '@context'      => 'https://www.w3.org/ns/activitystreams',
    'id'            => "$baseURL?page=$page",
    'type'          => 'OrderedCollectionPage',
    'partOf'        => $baseURL,
    'orderedItems'  => $items
];
if ($start + $perPage < $total)
    $pageObj['next'] = "$baseURL?page=" . ($page + 1);

echo json_encode($pageObj, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
debug_log("Output JSON for page $page complete");
